<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/21
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;

use Dingo\Api\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\AdminRoleUser;
use Modules\Admin\Entities\AdminUser;
use Modules\Admin\Http\Requests\UserRequest;
use Modules\Admin\Http\Service\UserService;
use Modules\Admin\Http\Transformer\AdminUserTransformer;
use Modules\Admin\Http\Utils\Helper;

class UserController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"用户"},
     *     summary="列表",
     *     path="api/user/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/User"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $list = AdminUser::paginate(request('per_page', null));
        return $this->response->paginator($list,new AdminUserTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"用户"},
     *     summary="添加",
     *     path="api/user/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"username","password","name","avatar","role_id",},
     *                 @OA\Property(
     *                     property="username",
     *                     type="string",
     *                     description="用户名",
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     type="string",
     *                     description="密码",
     *                 ),
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="avatar",
     *                     type="string",
     *                     description="头像",
     *                 ),
     *                 @OA\Property(
     *                     property="role_id",
     *                     type="string",
     *                     description="角色ID",
     *                 ),
     *
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *      )
     * )
     *
     */
    public function add(UserRequest $request){
        $params = $request->all([
            'username','password','name','avatar','role_id',
            'phone','entry_time','department_id','skill','job_name','contract_id',
        ]);

        #默认密码
        if(empty($params['password'])){
            $params['password'] = '123456';
        }

        $params['salt'] = Helper::generateRandomString();
        $params['password'] = password_hash(md5($params['password'].$params['salt']), PASSWORD_DEFAULT);

        #工号生成
        $params['job_number'] = UserService::createJobNumber();

        $user = AdminUser::create($params);

        return $this->response->item($user,new AdminUserTransformer(true));
    }

    /**
     * @OA\Get(
     *     tags={"用户"},
     *     summary="详情",
     *     path="api/user/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/User")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($id){
        $user = AdminUser::where('id',$id)->first();
        if(!$user){
            abort(500,'用户不存在');
        }

        return $this->response->item($user,new AdminUserTransformer());
    }

    /**
     * @OA\Put(
     *     tags={"用户"},
     *     summary="编辑",
     *     path="api/user/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"username","password","name","avatar","role_id",},
     *                 @OA\Property(
     *                     property="username",
     *                     type="string",
     *                     description="用户名",
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     type="string",
     *                     description="密码",
     *                 ),
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="avatar",
     *                     type="string",
     *                     description="头像",
     *                 ),
     *                 @OA\Property(
     *                     property="role_id",
     *                     type="string",
     *                     description="角色ID",
     *                 ),
     *
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(UserRequest $request,$id){
        $params = $request->all([
            'username','password','name','avatar','role_id',
            'phone','entry_time','department_id','skill','job_name','contract_id',
        ]);
        $user = AdminUser::where('id',$id)->first();

        if(!$user){
            abort(500,'用户不存在');
        }

        if($user->username != $params['username'] && AdminUser::where('username',$params['username'])->first()){
            abort(500,'用户名重复');
        }

        if(!empty($params['password'])){
            $params['password'] = password_hash(md5($params['password'].$user->salt), PASSWORD_DEFAULT);
        }else{
            unset($params['password']);
        }

        $user->update($params);

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"用户"},
     *     summary="删除",
     *     path="api/user/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($id){
        $user = AdminUser::where('id',$id)->first();
        if($user){
            $user->delete();
        }

        return $this->response->noContent()->statusCode(204);
    }
}