<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/21
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;

use Dingo\Api\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\AdminMenu;
use Modules\Admin\Entities\AdminRoleMenu;
use Modules\Admin\Http\Requests\MenuRequest;
use Modules\Admin\Http\Transformer\AdminMenuTransformer;
use Modules\Admin\Http\Utils\Helper;

class MenuController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"菜单"},
     *     summary="列表",
     *     path="api/menu/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Menu"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $menu = AdminMenu::orderBy('order','asc')->get();
        return $this->response->collection($menu,new AdminMenuTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"菜单"},
     *     summary="添加",
     *     path="api/menu/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string",
     *                      description="标题"
     *                 ),
     *                 @OA\Property(
     *                     property="icon",
     *                     type="string",
     *                      description="图标"
     *                 ),
     *                 @OA\Property(
     *                     property="key",
     *                     type="string",
     *                      description="标识"
     *                 ),
     *                 @OA\Property(
     *                     property="api",
     *                     type="string",
     *                      description="api地址"
     *                 ),
     *                 @OA\Property(
     *                     property="parent_id",
     *                     type="string",
     *                      description="父级菜单id"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *      )
     * )
     *
     */
    public function add(MenuRequest $request){
        $params = $request->all(['title','key','icon','api','is_menu','parent_id']);

        AdminMenu::create($params);

        return $this->response->noContent()->statusCode(201);
    }

    /**
     * @OA\Get(
     *     tags={"菜单"},
     *     summary="详情",
     *     path="api/menu/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Menu")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($id){
        $menu = AdminMenu::where('id',$id)->first();
        if(!$menu){
            abort(500,'菜单不存在');
        }

        return $this->response->item($menu,new AdminMenuTransformer());

    }

    /**
     * @OA\Put(
     *     tags={"菜单"},
     *     summary="编辑",
     *     path="api/menu/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="uri",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="parent_id",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(MenuRequest $request,$id){
        $params = $request->all(['title','key','icon','api','is_menu','parent_id']);
        $menu = AdminMenu::where('id',$id)->first();

        if(!$menu){
            abort(500,'菜单不存在');
        }

        if($menu->key != $params['key'] && AdminMenu::where('key',$params['key'])->first()){
            abort(500,'路由标识重复');
        }

        if($menu->id === $params['parent_id']){
            abort(500,'父级菜单不能选择自己');
        }

        $menu->update($params);

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Put(
     *     tags={"菜单"},
     *     summary="保存数据更改",
     *     path="api/menu/save_change",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"change_data"},
     *                 @OA\Property(
     *                     property="change_data",
     *                     description="变更数据",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function saveChange(Request $request){
        $params = $request->all(['change_data']);
        if(!is_array($params['change_data'])){
            abort(500,'数据错误');
        }

        $list = AdminMenu::whereIn('id',array_keys($params['change_data']))->get();
        if(!$list){
            abort(500,'数据错误');
        }

        DB::transaction(function ()use ($list,$params){
            $list->each(function ($menu)use ($params){
                $menu->update($params['change_data'][$menu->id]);
            });
        });

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"菜单"},
     *     summary="删除",
     *     path="api/menu/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($id){
        $menu = AdminMenu::where('id',$id)->first();
        if($menu){
            DB::transaction(function () use ($menu){
                $menu->delete();
                AdminRoleMenu::where('menu_id',$menu->id)->delete();

                #子菜单删除
                $descendantsIds = $menu->descendantsIds();
                AdminMenu::whereIn('id',$descendantsIds)->delete();
                AdminRoleMenu::whereIn('menu_id',$descendantsIds)->delete();
            });
        }
        return $this->response->noContent()->statusCode(204);
    }

}