<?php

namespace Modules\Admin\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Str;
use Kalnoy\Nestedset\NodeTrait;

class Area extends Model
{
    use NodeTrait;

    protected $table = 'area';

    protected $fillable = ['name', 'disabled', 'parent_id'];

    const CACHE_KEY_AREAS_ALL = 'package_area_all';
    const CACHE_KEY_AREAS_ALL_JS = 'package_area_all_js';


    /**
     * 清除缓存
     */
    public function clearCache()
    {
        Cache::forget(self::CACHE_KEY_AREAS_ALL);
        Cache::forget(self::CACHE_KEY_AREAS_ALL_JS);
    }

    /**
     * Settings constructor.
     *
     * @param array $attributes
     */

    /*
    public function __construct($attributes = [])
    {
        parent::__construct($attributes);

        $this->setConnection(config('admin.database.connection') ?: config('database.default'));

        $this->setTable(config('admin.extensions.config.table', 'area'));
    }

    */

    /**
     * @param $query
     *
     * @return QueryBuilder
     */
    public function scopeNotDeleted($query)
    {
        return $query->where('disabled', 0);
    }

    /**
     * 返回包含有从顶级到当前级的字符串组合
     *
     * @param string $implode
     *
     * @return mixed|string
     */
    public function getArea($implode = '')
    {
        if (!$this->id) {
            throw \Exception('需要实例化的类');
        }

        if (!$this->parent_id) {
            return $this->name;
        }

        return implode($implode, $this->ancestors->pluck('name')->toArray()).$implode.$this->name;
    }

    /**
     * 返回当前区域的ID&父ID,以便供选择的时候设置默认选项
     *
     * @param string $implode
     *
     * @return mixed|string
     */
    public function getAreaIds($implode = ',')
    {
        if (!$this->id) {
            throw \Exception('需要实例化的类');
        }

        if (!$this->parent_id) {
            return $this->id;
        }

        return implode($implode, $this->ancestors->pluck('id')->toArray()).$implode.$this->id;
    }

    /**
     * 返回地区列表
     *
     * @param null $parent_id
     *
     * @return mixed
     */
    public function getAreas($parent_id = null)
    {
        $area = $this->notDeleted()->withDepth();
        if ($parent_id) {
            $area->where('parent_id', $parent_id);
        } else {
            $area->whereNull('parent_id');
        }

        return $area->get();
    }

    public static function boot()
    {
        static::creating(function ($model) {
            $level = 0;
            if ($model->parent_id) {
                $level = static::withDepth()->whereKey($model->parent_id)->value('depth') + 1;
                $query = static::where('parent_id', $model->parent_id);
            } else {
                $query = static::whereNull('parent_id');
            }

            $step = pow(10, (3 - $level) * 2);

            $parent_code = 0;
            if ($model->parent_id) {
                $parent_code = intval(static::whereKey($model->parent_id)->value('code'));
            } else {
                $parent_code = 10000000;
            }

            //按每级 100 个计算
            $ids = (clone $query)->pluck('code')->toArray(); //同级下已用code
            $ids2 = range($parent_code + 1 * $step, $parent_code + 99 * $step, $step); //同级下所有code
            $diff = array_diff($ids2, $ids); //取出第一个可用的
            $code = array_first($diff);

            /*
            $prev = (clone $query)->max('code');
            //第一个
            if (!$prev && $model->parent_id) {
                $prev = $parent_code;
            }

            if (($prev + $step) > ($parent_code+100*$step) ) {
                $ids = (clone $query)->pluck('code')->toArray();//同级下所有code
                $ids2 = range(1*$step, 99*$step, $step );//同级下所有可用code
                $diff = array_diff($ids2, $ids);//取出一个没有用到的
                $code = array_first($diff);
            } else {
                $code = $prev + $step;
            }
            */

            $model->code = $code;
        });

        static::saved(function (Model $model){
            $model->clearCache();
        });
        parent::boot(); // TODO: Change the autogenerated stub
    }

    /**
     * 获得Root
     *
     * @return AreaModel
     */
    public function getRoot()
    {
        return $this->isRoot() ? $this : $this->ancestors->first(function ($area) {
            return $area->isRoot();
        });
    }

    public static function guessId($name, $parent_id = null)
    {
        if (!$name) {
            return null;
        }

        $name = Str::substr($name, 0, 2);

        return static::when($parent_id, function ($query, $parent_id) {
            $parent_id ? $query->where('parent_id', $parent_id) : $query->whereNull('parent_id');
        })->where('name', 'like', "{$name}%")->value('id');
    }

    public static function guessId2($names)
    {
        if (!is_array($names) || count($names) < 1) {
            return null;
        }

        $parent_id = null;
        $area_id = null;
        $i = 0;
        do {
            $area_id = static::guessId($names[$i], $parent_id);

            $i++;
            $area_id && $parent_id = $area_id;
        } while ($area_id && isset($names[$i]));

        return $parent_id;
    }

    /**
     * @param bool $model
     * @param int  $depth
     *
     * @return bool|null
     */
    public static function getAreaByIp($model = false, $depth = -1)
    {
        if (($return = static::getAreaByLocation($model, $depth)) !== false) {
            return $return;
        }

        $ip = \Request::ip();

        //检查是否是内网IP，是的话返回false
        $result = filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE);

        $ip_query = $result ? '&ip='.$ip : '';
        $data = Cache::rememberForever('baidu_ip_to_location_'.$ip, function () use ($ip_query){
            $url = 'http://api.map.baidu.com/location/ip?ak='.config('area.baidu_ak', '8d6687623e58af52fd73241ec01a224d').$ip_query;
            return @json_decode(file_get_contents($url), true);
        });

        $province = $data['content']['address_detail']['province'] ?? null;
        $city = $data['content']['address_detail']['city'] ?? null;
        $district = $data['content']['address_detail']['district'] ?? null;

        $querys = [$province, $city, $district];

        return self::getAreaByLocationString($querys, $model, $depth);
    }

    /**
     * @param array $querys 省市区三个字符串组成的数组
     * @param bool  $model  是返回查找到的ID还是对应模型
     * @param int   $depth  返回查找到的省市区级别，-1为最深，0为省，1为市，2为县
     *
     * @return null
     */
    public static function getAreaByLocationString($querys = [], $model = false, $depth = -1)
    {
        if ($depth > -1) {
            $querys = array_slice($querys, 0, ($depth + 1));
        }

        if (Str::substr($querys[0], 0, 2) == '北京') {
            $querys[0] = '北京';
        }elseif (Str::substr($querys[0], 0, 2) == '上海') {
            $querys[0] = '上海';
        }elseif (Str::substr($querys[0], 0, 2) == '重庆') {
            $querys[0] = '重庆';
        }elseif (Str::substr($querys[0], 0, 2) == '天津') {
            $querys[0] = '天津';
        }

        $id = self::guessId2($querys);

        if ($model) {
            return $id ? static::find($id) : $id;
        }

        return $id;
    }

    public static function getAreaByLocation($model = false, $depth = -1, $lat = null, $lng = null)
    {
        if (is_null($lat) && is_null($lng)) {
            $lat = request('lat', null);
            $lng = request('lng', null);
        }

        if (is_null($lat) || is_null($lng)) {
            return false;
        }

        $data = Cache::rememberForever('baidu_location_data_'.$lat.'_'.$lng, function () use ($lat, $lng){
            $query = "output=json&location={$lat},{$lng}";
            $url = 'http://api.map.baidu.com/reverse_geocoding/v3/?ak='.config('area.baidu_ak', '8d6687623e58af52fd73241ec01a224d').'&'.$query;
            return @json_decode(file_get_contents($url), true);
        });
        
        $province = $data['result']['addressComponent']['province'] ?? null;
        $city = $data['result']['addressComponent']['city'] ?? null;
        $district = $data['result']['addressComponent']['district'] ?? null;

        $querys = [$province, $city, $district];

        return self::getAreaByLocationString($querys, $model, $depth);
    }
}
