<?php

declare(strict_types=1);

namespace Modules\Admin\Http\Controllers;

use Exception;
use Dingo\Api\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\AdminMenu;
use Modules\Admin\Entities\AdminUser;
use Modules\Admin\Entities\Announcement;
use Modules\Admin\Events\UserLoginEvents;
use Modules\Admin\Http\Transformers\AccessTokenTransformer;
use Modules\Admin\Http\Transformers\AdminUserTransformer;
use Modules\Admin\Http\Transformers\BaseTransformer;
use Modules\Admin\Http\Utils\Helper;

class AuthController extends BaseController
{
    /**
     * @OA\Post(
     *     tags={"基础接口"},
     *     summary="登录",
     *     path="api/auth/login",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"username","password"},
     *                 @OA\Property(
     *                     property="username",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="登陆成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Token")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function login(Request $request)
    {
        $req      = $request->all(['username','password']);
        $username = $req["username"];
        $password = $req["password"];
        if ($username == "" || $password == "")
        {
            throw new \Exception("用户名或密码不能为空");
        }
        $user = AdminUser::where("username", $username)->first();
        if (!$user)
        {
            throw new \Exception("用户名或密码错误");
        }
        if (!password_verify(md5($password.$user->salt),$user->password))
        {
            throw new \Exception("用户名或密码错误");
        }

        event(new UserLoginEvents($user));

        return $this->response->item($user, new AccessTokenTransformer())->setStatusCode(201);
    }

    /**
     *
     * @OA\Put(
     *     tags={"基础接口"},
     *     summary="刷新token",
     *     path="api/auth/refresh",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="201",
     *          description="刷新成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Token")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function refreshToken(){
        return $this->response->item($this->user, new AccessTokenTransformer(true))->setStatusCode(201);
    }

    /**
     *
     * @OA\Put(
     *     tags={"基础接口"},
     *     summary="登出",
     *     path="api/auth/login_out",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="登出成功",
     *      )
     * )
     *
     */
    public function loginOut(){
        if($this->user){
            auth('api')->logout(true);
        }
        return $this->response->noContent()->statusCode('204');
    }

    public function userInfo(){
        return $this->response->item($this->user, new AdminUserTransformer());
    }

    /**
     *
     * @OA\Put(
     *     tags={"基础接口"},
     *     summary="获取配置",
     *     path="api/configs",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Menu")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function apiConfig(Request $request)
    {
        $user  = $this->user;
        $role = $user->role;

        $announcements = Announcement::canShow()->orderBy('order','desc')->orderBy('updated_at','desc')->get();

        $data = [
            'user_info' => Helper::mapAttr($user,['id', 'username', 'avatar', 'avatar_full_url', 'name', 'created_at', 'updated_at']),
            "is_admin"    => Helper::isAdministrator($role->id),
            "menus"    => Helper::getRoleMenu($role,true),
            "apis"    => Helper::getRoleApi($role,true),
            "announcements"    => $announcements,
        ];

        return $this->response->array(['data'=>$data]);
    }
}
