<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/31
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;

use Dingo\Api\Http\Request;
use Modules\Admin\Entities\Announcement;
use Modules\Admin\Http\Requests\AnnouncementRequest;
use Modules\Admin\Http\Transformers\AnnouncementTransformer;

class AnnouncementController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"公告"},
     *     summary="列表",
     *     path="api/announcement/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Announcement"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $list = Announcement::orderBy('order','desc')->orderBy('updated_at','desc')->paginate(request('per_page', null));
        return $this->response->paginator($list,new AnnouncementTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"公告"},
     *     summary="添加",
     *     path="api/announcement/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title","content"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string",
     *                     description="标题",
     *                 ),
     *                 @OA\Property(
     *                     property="content",
     *                     type="string",
     *                     description="内容",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Announcement")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function add(AnnouncementRequest $request){
        $params = array_filter($request->all(['title','content','show','top']));

        $announcement = Announcement::create($params);

        return $this->response->item($announcement,new AnnouncementTransformer());
    }

    /**
     * @OA\Get(
     *     tags={"公告"},
     *     summary="详情",
     *     path="api/announcement/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Announcement")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($id){
        $announcement = Announcement::where('id',$id)->first();
        if(!$announcement){
            abort(500,'公告不存在');
        }

        return $this->response->item($announcement,new AnnouncementTransformer());
    }

    /**
     * @OA\Put(
     *     tags={"公告"},
     *     summary="编辑",
     *     path="api/announcement/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title","content"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string",
     *                     description="标题",
     *                 ),
     *                 @OA\Property(
     *                     property="content",
     *                     type="string",
     *                     description="内容",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(AnnouncementRequest $request,$id){
        $params = array_filter($request->all(['title','content','show','top']));

        $announcement = Announcement::where('id',$id)->first();

        if(!$announcement){
            abort(500,'公告不存在');
        }
        $announcement->update($params);

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Put(
     *     tags={"公告"},
     *     summary="修改字段",
     *     path="api/announcement/{id}/change",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"type"},
     *                 @OA\Property(
     *                     property="type",
     *                     type="string",
     *                     description="字段，show=是否显示 top=置顶",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function change(Request $request,$id){
        $params = $request->all(['type']);

        $announcement = Announcement::where('id',$id)->first();
        if($announcement){
            switch ($params['type']){
                case "show":
                    $announcement->show = !$announcement->show;
                    break;
                case "top":
                    $max_order = Announcement::orderBy('order','desc')->value('order');
                    $announcement->order = $max_order + 1;
                    break;
            }
            $announcement->save();
        }

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"公告"},
     *     summary="删除",
     *     path="api/announcement/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($id){
        $announcement = Announcement::where('id',$id)->first();
        if($announcement){
            $announcement->delete();
        }

        return $this->response->noContent()->statusCode(204);
    }
}