<?php
declare(strict_types=1);

namespace Modules\Admin\Http\Utils;

use Carbon\Carbon;
use Illuminate\Support\Facades\Cache;
use Modules\Admin\Entities\AdminMenu;
use Modules\Admin\Entities\AdminUser;

class Helper
{

    /**
     * 获取全部菜单
     * @desc
     * @return mixed|mixed[]
     * @since     2023/10/23
     * @modify
     * @author    [ZZM]
     */
    static function getAllMenu($refresh = FALSE)
    {
        if (!$refresh && $menu = Cache::get("ALL_MENU"))
        {
            return json_decode($menu, TRUE);
        }
        $menu = AdminMenu::orderBy('order','asc')->get()->toArray();
        if ($menu)
        {
            Cache::put("ALL_MENU", json_encode($menu));
        }

        return $menu;
    }

    /**
     * 获取全部接口地址
     * @desc
     *
     * @param false $refresh
     *
     * @return mixed|mixed[]
     * @author    [ZZM]
     * @since     2023/10/30
     * @modify
     */
    static function getAllApi($refresh = FALSE)
    {
        if (!$refresh && $apis = Cache::get("ALL_API"))
        {
            return json_decode($apis, TRUE);
        }
        $apis = AdminMenu::all()->toArray();
        if ($apis)
        {
            Cache::put("ALL_API", json_encode($apis));
        }

        return $apis;
    }

    /**
     * 获取角色菜单
     * @desc
     *
     * @param $role
     *
     * @return mixed|mixed[]
     * @author    [ZZM]
     * @since     2023/10/23
     * @modify
     */
    static function getRoleMenu($role, $refresh = FALSE)
    {
        if (Helper::isAdministrator($role->id))
        {
            #超级管理员默认获取全部菜单
            return self::getAllMenu(TRUE);
        }

        if (!$refresh && $menu = Cache::get("ROLE_MENU_{$role->id}"))
        {
            return json_decode($menu, TRUE);
        }

        $menu_ids = $role->menus?:[];
        if($menu_ids){
            $menu = AdminMenu::orderBy('order','asc')->whereIn('id',$menu_ids)->where('is_menu', 1)->get();
            $menu = $menu->toArray();
        }else{
            $menu = [];
        }

        if ($menu)
        {
            Cache::put("ROLE_MENU_{$role->id}", json_encode($menu));
        }

        return $menu;
    }

    /**
     * 获取角色接口地址
     * @desc
     *
     * @param       $role
     * @param false $refresh
     *
     * @return mixed|mixed[]
     * @since     2023/10/30
     * @modify
     * @author    [ZZM]
     */
    static function getRoleApi($role, $refresh = FALSE)
    {
        if (Helper::isAdministrator($role->id))
        {
            #超级管理员默认获取全部菜单
            return self::getAllApi($refresh);
        }

        if (!$refresh && $apis = Cache::get("ROLE_API_{$role->id}"))
        {
            return json_decode($apis, TRUE);
        }

        //$apis = $role->menu()->get();
        $menu_ids = $role->menus?:[];
        if($menu_ids){
            $apis = AdminMenu::whereIn('id',$menu_ids)->get();
            $apis = $apis->toArray();
        }else{
            $apis = [];
        }

        if ($apis)
        {
            Cache::put("ROLE_API_{$role->id}", json_encode($apis));
        }

        return $apis;
    }

    /**
     * 是否超级管理员
     * @desc
     *
     * @param $role_id
     *
     * @return bool
     * @author    [ZZM]
     * @since     2023/10/23
     * @modify
     */
    static function isAdministrator($role_id)
    {
        return $role_id == 1;
    }

    static function makeTree(array $elements, $parentId = 0)
    {
        $branch = [];

        foreach ($elements as $element)
        {
            if ($element['parent_id'] == $parentId)
            {
                $children = self::makeTree($elements, $element['id']);
                if ($children)
                {
                    $element['children'] = $children;
                }
                $branch[] = $element;
            }
        }

        return $branch;
    }

    /**
     * 随机字符串
     * @desc
     *
     * @param int $length
     *
     * @return string
     * @author    [ZZM]
     * @since     2023/10/21
     * @modify
     */
    static function generateRandomString($length = 12)
    {
        $characters   = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*()-_+=<>?';
        $charLength   = strlen($characters);
        $randomString = '';

        for ($i = 0; $i < $length; $i++)
        {
            $randomString .= $characters[ rand(0, $charLength - 1) ];
        }

        return $randomString;
    }

    static function mapAttr($obj, $keys = [])
    {
        $request        = request();
        $temp           = [];
        $fields         = $request->get('fields', '');
        $fields         = $fields == '' ? [] : explode(',', $fields);
        $exclude_fields = $request->get('exclude_fields', '');
        $exclude_fields = $exclude_fields == '' ? [] : explode(',', $exclude_fields);

        foreach ($keys as $index => $key)
        {
            if (!is_numeric($index))
            {
                if (is_string($key) && \Illuminate\Support\Str::startsWith($key, '_call:'))
                {
                    [$class, $method, $arg_str] = explode('@', substr($key, 6) . '@@');
                    $args = explode(',', $arg_str);
                    array_unshift($args, $obj);
                    $temp[ $index ] = (new $class())->$method(...$args);
                }
                else
                {
                    $temp[ $index ] = $key;
                }
            }
            else
            {
                if (strpos($key, '.'))
                {
                    [$relation, $attr] = explode('.', $key);
                    $temp[ $key ] = optional($obj->$relation)->$attr;
                }
                else
                {
                    if (!empty($fields) && !in_array($key, $fields))
                    {
                        continue;
                    }

                    if (in_array($key, $exclude_fields))
                    {
                        continue;
                    }

                    if (is_string($key) && \Illuminate\Support\Str::endsWith($key, '_full_url'))
                    {
                        $key2         = substr($key, 0, -9);
                        $temp[ $key ] = $obj->$key ?? self::fileUrl($obj->$key2);
                    }
                    else
                    {
                        $temp[ $key ] = (is_object($obj->$key) && ($obj->$key instanceof Carbon)) ? $obj->$key->toDateTimeString() : $obj->$key;
                    }
                }
            }
        }

        return $temp;
    }

    static function getFileUrl($path, $type)
    {
        $file_config = config('upload.' . $type, []);
        $key         = 'fileststems.disks.' . $file_config['filesystem'] . '.url';
        $prefix      = $file_config['url'] ?? config($key, NULL);

        return $prefix ? $prefix . $path : NULL;
    }

    static function fileUrl($file_path)
    {
        if ($file_path == NULL)
        {
            return NULL;
        }

        if (is_array($file_path))
        {
            return array_map('Helper::fileUrl', $file_path);
        }

        if (substr($file_path, 0, 4) == 'http')
        {
            return $file_path;
        }

        $host_prefix = config('app.file_url', config('app.url', ''));

        if (substr($file_path, 0, 1) == '/')
        {
            return $host_prefix . $file_path;
        }

        return $host_prefix . config('app.file_prefix', '/uploads/') . $file_path;
    }
}
