<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/31
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;


use Modules\Admin\Entities\Department;
use Modules\Admin\Http\Requests\DepartmentRequest;
use Modules\Admin\Http\Transformers\DepartmentTransformer;
use Modules\Admin\Http\Utils\Helper;

class DepartmentController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"部门"},
     *     summary="列表",
     *     path="api/department/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Department"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $list = Department::paginate(request('per_page', null));
        return $this->response->paginator($list,new DepartmentTransformer());

    }

    /**
     * @OA\Get(
     *     tags={"部门"},
     *     summary="数据列表",
     *     path="api/department/fetch_list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Department"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function dataList(){
        $roles = Department::all();
        return $this->response->collection($roles,new DepartmentTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"部门"},
     *     summary="添加",
     *     path="api/department/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","desc"},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="desc",
     *                     type="string",
     *                     description="说明",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Department")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function add(DepartmentRequest $request){
        $params = $request->all(['name','desc']);

        $department = Department::create($params);

        return $this->response->item($department,new DepartmentTransformer());
    }

    /**
     * @OA\Get(
     *     tags={"部门"},
     *     summary="详情",
     *     path="api/department/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Department")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($id){
        $department = Department::where('id',$id)->first();
        if(!$department){
            abort(500,'部门不存在');
        }

        return $this->response->item($department,new DepartmentTransformer());
    }

    /**
     * @OA\Put(
     *     tags={"部门"},
     *     summary="编辑",
     *     path="api/department/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","desc"},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="desc",
     *                     type="string",
     *                     description="说明",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(DepartmentRequest $request,$id){
        $params = $request->all(['name','desc']);
        $department = Department::where('id',$id)->first();

        if(!$department){
            abort(500,'部门不存在');
        }
        $department->update($params);

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"部门"},
     *     summary="删除",
     *     path="api/department/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($id){
        $department = Department::where('id',$id)->first();
        if($department){
            $department->delete();
        }

        return $this->response->noContent()->statusCode(204);
    }
}