<?php

namespace Modules\Customers\Http\Controllers;

use Illuminate\Support\Carbon;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\Area;
use Modules\Customers\Entities\Customer;
use Modules\Customers\Entities\CustomerFollow;
use Modules\Customers\Http\Requests\CustomerRequest;
use Modules\Customers\Http\Service\CustomerService;
use Modules\Customers\Http\Transformer\CustomerTransformer;

class CustomersController extends BaseController
{
    protected $type;

    /**
     * 客户列表
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function list(CustomerRequest $request)
    {
        $params = $request->input('search',[]);

        $list = Customer::{$this->type}()->paramsSearch($params)->paginate(request('per_page'));

        return $this->response->paginator($list, new CustomerTransformer());

    }

    /**
     * 新增客户
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function add(CustomerRequest $request)
    {
        $params = $request->only(
            [
                'name', 'company', 'industry', 'department', 'duties', 'tel_phone', 'mobile_phone',
                'wechat', 'dy', 'email', 'province', 'city', 'area', 'address', 'follow_status',
                'from', 'type','business_license', 'identity', 'bank_account',
            ]
        );

        $params = array_filter($params);

        $params['founder'] = $this->user->id;
        if($this->type == 'private'){
            $params['owner']   = $this->user->id;
        }

        $customer = Customer::create($params)->refresh();

        return $this->response->item($customer, new CustomerTransformer());
    }

    /**
     * 批量导入客户
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function addImport(CustomerRequest $request)
    {
        $params = $request->input('data', []);

        if (empty($params))
        {
            abort('500', '内容为空');
        }

        $data = CustomerService::formatImportData($params, $this->type);

        Customer::insert($data);

        return $this->response->created();
    }

    /**
     * 客户详情
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function show(CustomerRequest $request, $id)
    {
        $customer = Customer::where('id', $id)->{$this->type}()->first();
        if (!$customer)
        {
            abort(500, '客户不存在');
        }

        return $this->response->item($customer, new CustomerTransformer());
    }

    /**
     * 编辑 普通资料
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function edit(CustomerRequest $request, $id)
    {
        $params   = $request->only(
            [
                'name', 'company', 'industry', 'department', 'duties', 'tel_phone', 'mobile_phone',
                'wechat', 'dy', 'email', 'province', 'city', 'area', 'address', 'follow_status',
                'from', 'type', 'founder', 'owner',
            ]
        );
        $customer = Customer::where('id', $id)->{$this->type}()->first();
        if (!$customer)
        {
            abort(500, '客户不存在');
        }

        $update = CustomerService::formatUpdateParams($customer, $params);

        $customer->update($update);

        return $this->response->item($customer, new CustomerTransformer());
    }

    /**
     * 编辑 私密资料
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function editPrivate(CustomerRequest $request, $id)
    {
        $params   = $request->only(
            [
                'business_license', 'identity', 'bank_account','mobile_phone',
            ]
        );

        $customer = Customer::where('id', $id)->{$this->type}()->first();
        if (!$customer)
        {
            abort(500, '客户不存在');
        }

        $update = CustomerService::formatUpdateParams($customer, $params);

        $customer->update($update);

        return $this->response->item($customer, new CustomerTransformer());
    }

    /**
     * 更新跟进状态
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function updateFollowStatus(CustomerRequest $request, $id)
    {
        $params   = $request->only(['type', 'status', 'next_follow_time', 'content']);
        $customer = Customer::where('id', $id)->{$this->type}()->first();
        if (!$customer)
        {
            abort(500, '客户不存在');
        }

        DB::transaction(function ()use ($customer,$params){
            $params['customer_id'] = $customer->id;
            $params['before_status'] = $customer->follow_status;
            $params['follow_time'] = Carbon::now();
            CustomerFollow::create($params);

            $customer->follow_status = $params['status'];
            $customer->save();
        });

        return $this->response->noContent();
    }

    /**
     * 数据字典
     * @desc
     * @author    [ZZM]
     * @since     2023/11/2
     * @modify
     */
    public function dataMaps(){
        $type_maps = Customer::$type_maps;
        $type_maps = array_map(function ($id, $name) {
            return ['id' => $id, 'name' => $name];
        }, array_keys($type_maps), $type_maps);

        $from_maps = Customer::$from_maps;
        $from_maps = array_map(function ($id, $name) {
            return ['id' => $id, 'name' => $name];
        }, array_keys($from_maps), $from_maps);

        $follow_status_maps = CustomerFollow::$status_maps;
        $follow_status_maps = array_map(function ($id, $name) {
            return ['id' => $id, 'name' => $name];
        }, array_keys($follow_status_maps), $follow_status_maps);

        $follow_type_maps = CustomerFollow::$type_maps;
        $follow_type_maps = array_map(function ($id, $name) {
            return ['id' => $id, 'name' => $name];
        }, array_keys($follow_type_maps), $follow_type_maps);

        $data = [
            'type_maps' => $type_maps,
            'from_maps' => $from_maps,
            'follow_status_maps' => $follow_status_maps,
            'follow_type_maps' => $follow_type_maps,
        ];
        return $this->response->array(['data'=>$data]);
    }
}
