<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/21
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;

use Dingo\Api\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\AdminMenu;
use Modules\Admin\Entities\AdminRole;
use Modules\Admin\Entities\AdminRoleMenu;
use Modules\Admin\Entities\AdminRoleUser;
use Modules\Admin\Http\Requests\RoleRequest;
use Modules\Admin\Http\Transformers\AdminRoleTransformer;

class RoleController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"角色"},
     *     summary="列表",
     *     path="api/role/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Role"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $roles = AdminRole::all();
        return $this->response->collection($roles,new AdminRoleTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"角色"},
     *     summary="添加",
     *     path="api/role/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","slug",},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="slug",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *      )
     * )
     *
     */
    public function add(RoleRequest $request){
        $params = $request->all(['name','slug']);

        AdminRole::create($params);

        return $this->response->noContent()->statusCode(201);
    }

    /**
     * @OA\Get(
     *     tags={"角色"},
     *     summary="详情",
     *     path="api/role/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Role")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($role_id){
        $role = AdminRole::where('id',$role_id)->first();
        if(!$role){
            abort(500,'角色不存在');
        }

        return $this->response->item($role,new AdminRoleTransformer(true));
    }

    /**
     * @OA\Put(
     *     tags={"角色"},
     *     summary="编辑",
     *     path="api/role/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","slug",},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="slug",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(RoleRequest $request,$role_id){
        $params = $request->all(['name','slug','menus']);
        $role = AdminRole::where('id',$role_id)->first();

        if(!$role){
            abort(500,'角色不存在');
        }

        if(!is_array($params['menus'])){
            abort(422,'菜单格式错误');
        }

        if($role_id == 1){
            abort(500,'超级管理员不可编辑权限');
        }

        $menus = AdminMenu::all()->toArray();
        if($error = $this->checkMenu($menus,$params['menus'])){
            abort(500,$error);
        }

        DB::transaction(function () use ($role,$params,$role_id){
            $role->name = $params['name'];
            $role->slug = $params['slug'];
            $role->save();

            AdminRoleMenu::where('role_id',$role_id)->delete();

            $insert = [];
            foreach ($params['menus'] as $item){
                $insert[] = [
                    'role_id' => $role_id,
                    'menu_id' => (int)$item,
                ];
            }
            if($insert){
                AdminRoleMenu::insert($insert);
            }
        });
        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"角色"},
     *     summary="删除",
     *     path="api/role/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($role_id){
        $role = AdminRole::where('id',$role_id)->first();


        if($role){
            if(AdminRoleUser::where('role_id',$role->id)->count() > 0){
                abort("此角色还有关联的用户，无法删除");
            }

            DB::transaction(function () use ($role){
                $role->delete();
                AdminRoleMenu::where('role_id',$role->id)->delete();
            });
        }

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * 菜单格式是否正确
     * @desc
     *
     * @param $menus
     * @param $input_menu_ids
     *
     * @return string
     * @since     2023/10/21
     * @modify
     * @author    [ZZM]
     */
    function checkMenu($menus,$input_menu_ids){
        $menu_ids = array_column($menus,'id');
        $menu_parent_ids = array_column($menus,'parent_id','id');
        foreach ($input_menu_ids as $id){
            #不存在的菜单
            if(!in_array($id,$menu_ids)){
                return '包含不存在的菜单';
            }

            #选择了子菜单，但是没包含父级菜单
            $parent_id = $menu_parent_ids[$id];
            if($parent_id > 0 && !in_array($parent_id,$input_menu_ids)){
                return '未包含父级菜单';
            }
        }

        return '';
    }
}