<?php
/**
 * @desc
 * @author    [ZZM]
 * @since     2023/10/21
 * @copyright
 */

namespace Modules\Admin\Http\Controllers;

use Dingo\Api\Http\Request;
use Illuminate\Support\Facades\DB;
use Modules\Admin\Entities\AdminMenu;
use Modules\Admin\Entities\AdminRoleMenu;
use Modules\Admin\Http\Requests\MenuRequest;
use Modules\Admin\Http\Transformers\AdminMenuTransformer;
use Modules\Admin\Http\Utils\Helper;

class MenuController extends BaseController
{
    /**
     * @OA\Get(
     *     tags={"菜单"},
     *     summary="列表",
     *     path="api/menu/list",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Menu"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     *
     */
    public function list(){
        $menu = AdminMenu::all();
        return $this->response->collection($menu,new AdminMenuTransformer());
    }

    /**
     * @OA\Post(
     *     tags={"菜单"},
     *     summary="添加",
     *     path="api/menu/add",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="uri",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="parent_id",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="",
     *      )
     * )
     *
     */
    public function add(MenuRequest $request){
        $params = $request->all(['title','uri','parent_id']);

        AdminMenu::create($params);

        return $this->response->noContent()->statusCode(201);
    }

    /**
     * @OA\Get(
     *     tags={"菜单"},
     *     summary="详情",
     *     path="api/menu/{id}/detail",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Menu")
     *              )
     *          )
     *      )
     * )
     *
     */
    public function detail($id){
        $menu = AdminMenu::where('id',$id)->first();
        if(!$menu){
            abort(500,'菜单不存在');
        }

        return $this->response->item($menu,new AdminMenuTransformer());

    }

    /**
     * @OA\Put(
     *     tags={"菜单"},
     *     summary="编辑",
     *     path="api/menu/{id}/edit",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"title"},
     *                 @OA\Property(
     *                     property="title",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="uri",
     *                     type="string"
     *                 ),
     *                 @OA\Property(
     *                     property="parent_id",
     *                     type="string"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function edit(MenuRequest $request,$id){
        $params = $request->all(['title','uri','parent_id']);
        $menu = AdminMenu::where('id',$id)->first();

        if(!$menu){
            abort(500,'菜单不存在');
        }

        if($params['uri'] && $menu->uri != $params['uri'] && AdminMenu::where('uri',$params['uri'])->first()){
            abort(500,'uri重复');
        }

        $menu->update($params);

        return $this->response->noContent()->statusCode(204);
    }

    /**
     * @OA\Delete(
     *     tags={"菜单"},
     *     summary="删除",
     *     path="api/menu/{id}/delete",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     *
     */
    public function delete($id){
        $menu = AdminMenu::where('id',$id)->first();

        if($menu){
            DB::transaction(function () use ($menu){
                $menu->delete();

                AdminRoleMenu::where('menu_id',$menu->id)->delete();
            });
        }
        return $this->response->noContent()->statusCode(204);
    }

}