import { login, logout, getInfo, getConfigs } from '@/api/auth'
import { getToken, setToken, removeToken } from '@/utils/auth'
import { resetRouter } from '@/router'

const getDefaultState = () => {
  return {
    token: getToken(),
    name: '',
    avatar: '',
    user_info: null,
    announcements: null,
    menus: null,
    is_admin: false
  }
}

const state = getDefaultState()

const mutations = {
  RESET_STATE: (state) => {
    Object.assign(state, getDefaultState())
  },
  SET_TOKEN: (state, token) => {
    state.token = token
  },
  SET_USER_INFO: (state, user_info) => {
    state.user_info = user_info
  },
  SET_NAME: (state, name) => {
    state.name = name
  },
  SET_AVATAR: (state, avatar) => {
    state.avatar = avatar
  },
  SET_ANNOUNCEMENTS: (state, announcements) => {
    state.announcements = announcements
  },
  SET_MENUS: (state, menus) => {
    state.menus = menus
  },
  SET_IS_ADMIN: (state, is_admin) => {
    state.is_admin = is_admin
  }
}

const actions = {
  // user login
  login({ commit }, userInfo) {
    const { username, password } = userInfo
    return new Promise((resolve, reject) => {
      login({ username: username.trim(), password: password }).then(response => {
        const { data } = response
        commit('SET_TOKEN', data.access_token)
        setToken(data.access_token)
        resolve()
      }).catch(error => {
        reject(error)
      })
    })
  },

  // get user info
  getInfo({ commit, state }) {
    return new Promise((resolve, reject) => {
      getInfo(state.token).then(response => {
        const { data } = response

        if (!data) {
          return reject('Verification failed, please Login again.')
        }

        const { name, avatar_full_url } = data

        commit('SET_NAME', name)
        commit('SET_AVATAR', avatar_full_url)
        resolve(data)
      }).catch(error => {
        reject(error)
      })
    })
  },

  getConfigs({ commit, state }) {
    return new Promise((resolve, reject) => {
      getConfigs().then(response => {
        const { data } = response

        if (!data) {
          return reject('获取配置失败，请重新登录')
        }

        const { user_info, announcements, menus, is_admin } = data

        commit('SET_USER_INFO', user_info)
        commit('SET_NAME', user_info.name)
        commit('SET_AVATAR', user_info.avatar_full_url)
        commit('SET_MENUS', menus)
        commit('SET_ANNOUNCEMENTS', announcements)
        commit('SET_IS_ADMIN', is_admin)

        resolve(data)
      }).catch(error => {
        reject(error)
      })
    })
  },

  // user logout
  logout({ commit, state }) {
    return new Promise((resolve, reject) => {
      logout(state.token).then(() => {
        removeToken() // must remove  token  first
        resetRouter()
        commit('RESET_STATE')
        resolve()
      }).catch(error => {
        reject(error)
      })
    })
  },

  // remove token
  resetToken({ commit }) {
    return new Promise(resolve => {
      removeToken() // must remove  token  first
      commit('RESET_STATE')
      resolve()
    })
  }
}

export default {
  namespaced: true,
  state,
  mutations,
  actions
}

