<?php

namespace Modules\Share\Http\Controllers\Api;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Modules\Share\Http\Requests\SchemeRequest;
use Modules\Share\Entities\scheme;
use Modules\Share\Entities\schemeConfig;
use Modules\Share\Services\SchemeService;
use Modules\Share\Transformers\SchemeConfigTransformer;
use Modules\Share\Transformers\SchemeTransformer;

class SchemeController extends ApiController
{
    /**
     * @return mixed
     *
     * @OA\Post(
     *     tags={"奖励方案"},
     *     summary="方案列表",
     *     path="/api/share/schemes/list",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"status"},
     *                 @OA\Property(
     *                     property="status",
     *                     type="integer",
     *                     description="要筛选的状态 0禁用 1启用",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Scheme"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index(Request $request)
    {
        $schemes = scheme::query()->where('status', $request->status)->paginate(20);
        return $this->response->paginator($schemes, new SchemeTransformer());

    }

    /**
     * 新增方案
     * @return mixed
     *
     * @OA\Post(
     *     tags={"奖励方案"},
     *     summary="新增方案",
     *     path="/api/share/schemes",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","direct_source_ratio","indirect_source_ratio","recharge_amount","consume_amount","unconsume_amount","online_pay","dividend_type","benefit_month","interests_images"},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="direct_source_ratio",
     *                     type="integer",
     *                     description="直接客源奖励比例",
     *                 ),
     *                 @OA\Property(
     *                     property="indirect_source_ratio",
     *                     type="integer",
     *                     description="间接客源奖励比例",
     *                 ),
     *                 @OA\Property(
     *                     property="recharge_amount",
     *                     type="float",
     *                     description="充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="consume_amount",
     *                     type="float",
     *                     description="可消费充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="unconsume_amount",
     *                     type="float",
     *                     description="不可消费充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="online_pay",
     *                     type="integer",
     *                     description="是否线上支付 0否 1是",
     *                 ),
     *                 @OA\Property(
     *                     property="dividend_type",
     *                     type="integer",
     *                     description="分红方式 1毛利分红 2实收分红",
     *                 ),
     *                 @OA\Property(
     *                     property="benefit_month",
     *                     type="integer",
     *                     description="受益时长",
     *                 ),
     *                 @OA\Property(
     *                     property="interests_images",
     *                     type="integer",
     *                     description="方案权益图片",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function create(SchemeRequest $request,SchemeService $schemeService)
    {
        $schemeService->create($request->all());

        return $this->response->created();
    }

    /**
     * 查看方案信息
     * @return mixed
     *
     * @OA\Get(
     *     tags={"奖励方案"},
     *     summary="查看方案信息",
     *     path="/api/share/schemes/{scheme_id}",
     *     @OA\Parameter(
     *         name="scheme_id",
     *         in="path",
     *         description="方案ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="object",
     *                      ref="#/components/schemas/Scheme",
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function show($scheme_id)
    {

        $scheme = scheme::query()->findOrFail($scheme_id);

        return $this->response->item($scheme, new SchemeTransformer());
    }

    /**
     * 修改方案
     * @return mixed
     *
     * @OA\Put(
     *     tags={"奖励方案"},
     *     summary="修改方案",
     *     path="/api/share/schemes",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"id","name","direct_source_ratio","indirect_source_ratio","recharge_amount","consume_amount","unconsume_amount","online_pay","dividend_type","benefit_month","interests_images"},
     *                 @OA\Property(
     *                     property="id",
     *                     type="integer",
     *                     description="方案Id",
     *                 ),
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="名称",
     *                 ),
     *                 @OA\Property(
     *                     property="direct_source_ratio",
     *                     type="integer",
     *                     description="直接客源奖励比例",
     *                 ),
     *                 @OA\Property(
     *                     property="indirect_source_ratio",
     *                     type="integer",
     *                     description="间接客源奖励比例",
     *                 ),
     *                 @OA\Property(
     *                     property="recharge_amount",
     *                     type="float",
     *                     description="充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="consume_amount",
     *                     type="float",
     *                     description="可消费充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="unconsume_amount",
     *                     type="float",
     *                     description="不可消费充值金额",
     *                 ),
     *                 @OA\Property(
     *                     property="online_pay",
     *                     type="integer",
     *                     description="是否线上支付 0否 1是",
     *                 ),
     *                 @OA\Property(
     *                     property="dividend_type",
     *                     type="integer",
     *                     description="分红方式 1毛利分红 2实收分红",
     *                 ),
     *                 @OA\Property(
     *                     property="benefit_month",
     *                     type="integer",
     *                     description="受益时长",
     *                 ),
     *                 @OA\Property(
     *                     property="interests_images",
     *                     type="integer",
     *                     description="方案权益图片",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="object",
     *                      ref="#/components/schemas/Scheme",
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function edit(SchemeRequest $request,SchemeService $schemeService)
    {
        $scheme = scheme::find($request->id);
        if(!$scheme){
            $this->response->errorNotFound();
        }

        $schemeService->update($request->id, $request->all());

        return $this->response->item($schemeService->getModel(), new SchemeTransformer());
    }

    /**
     * 删除方案
     * @return mixed
     *
     * @OA\Delete(
     *     tags={"奖励方案"},
     *     summary="删除方案",
     *     path="/api/share/schemes",
     *     @OA\Parameter(
     *         name="scheme_id",
     *         in="path",
     *         description="方案ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function destroy($scheme_id)
    {
        scheme::query()->where('id',$scheme_id)->delete();

        return $this->response->noContent();
    }

    /**
     * 启用/禁用方案
     * @return mixed
     *
     * @OA\Put(
     *     tags={"奖励方案"},
     *     summary="启用/禁用 方案",
     *     path="/api/share/schemes/status/{scheme_id}",
     *     @OA\Parameter(
     *         name="scheme_id",
     *         in="path",
     *         description="方案ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"status"},
     *                 @OA\Property(
     *                     property="status",
     *                     type="integer",
     *                     description="状态 0禁用 1启用",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="object",
     *                      ref="#/components/schemas/Scheme",
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function status($scheme_id,Scheme $schemeModel){
        $scheme = $schemeModel->find($scheme_id);
        if(!$scheme){
            $this->response->errorNotFound();
        }

        $scheme->status = request('status', 0)?1:0;
        $scheme->save();

        return $this->response->noContent();
    }

    /**
     * 查看方案配置
     * @return mixed
     *
     * @OA\Get(
     *     tags={"奖励方案"},
     *     summary="查看方案配置",
     *     path="/api/share/schemes_setting",
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="object",
     *                      ref="#/components/schemas/SchemeConfig",
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function showSetting(SchemeConfig $schemeConfigModel){
        $schemeConfig = $schemeConfigModel->first();

        return $this->response->item($schemeConfig, new SchemeConfigTransformer());
    }

    /**
     * 修改方案配置
     * @return mixed
     *
     * @OA\Put(
     *     tags={"奖励方案"},
     *     summary="修改方案配置",
     *     path="/api/share/schemes_setting",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"introduce_manager_ratio","default_manager_ratio"},
     *                 @OA\Property(
     *                     property="introduce_manager_ratio",
     *                     type="integer",
     *                     description="介绍掌柜奖励比例",
     *                 ),
     *                 @OA\Property(
     *                     property="default_manager_ratio",
     *                     type="integer",
     *                     description="默认奖励比例",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function setting(Request $request,SchemeConfig $schemeConfigModel){
        $request->validate([
            'introduce_manager_ratio' => 'required|Numeric|between:0,100',
            'default_manager_ratio' => 'required|Numeric|between:0,100',
        ]);

        $schemeConfig = $schemeConfigModel->first();
        $schemeConfig->introduce_manager_ratio = $request->introduce_manager_ratio;
        $schemeConfig->default_manager_ratio = $request->default_manager_ratio;
        $schemeConfig->save();

        return $this->response->noContent();
    }
}
