<?php

namespace Modules\Project\Providers;

use Illuminate\Database\Eloquent\Factory;
use Illuminate\Support\Facades\Event;
use Illuminate\Support\ServiceProvider;
use Modules\Project\Services\MacroService;
use Modules\Project\Services\UserObserve;

class ProjectServiceProvider extends ServiceProvider
{
    /**
     * Indicates if loading of the provider is deferred.
     *
     * @var bool
     */
    protected $defer = false;

    /**
     * Boot the application events.
     *
     * @param MacroService $macroService
     *
     * @return void
     */
    public function boot(MacroService $macroService)
    {
        $this->registerTranslations();
        $this->registerConfig();
        $this->registerViews();
        $this->registerFactories();
        $this->loadMigrationsFrom(__DIR__.'/../Database/Migrations');

        //添加一系列宏
        $macroService->addMacro();

        $userClass = config('auth.providers.users.model', 'App\\User');
        $userClass::observe(UserObserve::class);

        \Validator::extend('check_pay_password', function ($attribute, $value, $parameters, $validator) {
            $hashed_password = empty($parameters) ? \Auth::user()->wallet->pay_password : $parameters[0];

            return \Hash::check($value, $hashed_password);
        });
        \Validator::replacer('check_pay_password', function ($message, $attribute, $rule, $parameters) {
            return '支付密码错误';
        });

        config(['wechat.payment.default' => [
            'sandbox' => env('WECHAT_PAYMENT_SANDBOX', false),
            'app_id' => db_config('app_id', ''),
            'mch_id' => db_config('mch_id', ''),
            'key' => db_config('key', ''),
            'notify_url' => env('APP_URL', false).'/api/payment/notify',
        ]]);

        config(['pay.wechat'=> config('wechat.payment.default')]);

        (new \Overtrue\LaravelWeChat\ServiceProvider(app()))->register() ;

        $this->addPayExtends();
    }


    public function addPayExtends()
    {
        Event::listen('Modules\Base\Events\EnrollmentCreated','Modules\Project\Listeners\EnrollmentCreatedListener');
        Event::listen('SMG\\Pay\\Events\\PaymentSuccessEvent', 'Modules\Project\Listeners\PaymentSuccessListener');
    }

    /**
     * Register the service provider.
     *
     * @return void
     */
    public function register()
    {
        //
    }

    /**
     * Register config.
     *
     * @return void
     */
    protected function registerConfig()
    {
        $this->publishes([
            __DIR__.'/../Config/config.php' => config_path('project.php'),
        ], 'config');
        $this->mergeConfigFrom(
            __DIR__.'/../Config/config.php',
            'project'
        );
    }

    /**
     * Register views.
     *
     * @return void
     */
    public function registerViews()
    {
        $viewPath = resource_path('views/modules/project');

        $sourcePath = __DIR__.'/../Resources/views';

        $this->publishes([
            $sourcePath => $viewPath,
        ], 'views');

        $this->loadViewsFrom(array_merge(array_map(function ($path) {
            return $path.'/modules/project';
        }, \Config::get('view.paths')), [$sourcePath]), 'project');
    }

    /**
     * Register translations.
     *
     * @return void
     */
    public function registerTranslations()
    {
        $langPath = resource_path('lang/modules/project');

        if (is_dir($langPath)) {
            $this->loadTranslationsFrom($langPath, 'project');
        } else {
            $this->loadTranslationsFrom(__DIR__.'/../Resources/lang', 'project');
        }
    }

    /**
     * Register an additional directory of factories.
     *
     * @return void
     */
    public function registerFactories()
    {
        if (!app()->environment('production')) {
            app(Factory::class)->load(__DIR__.'/../Database/factories');
        }
    }

    /**
     * Get the services provided by the provider.
     *
     * @return array
     */
    public function provides()
    {
        return [];
    }
}
