<?php

namespace Modules\Project\Http\Controllers\Api;

use Modules\Base\Http\Controllers\Api\ApiController;
use Modules\Project\Http\Requests\WalletRequest;
use Modules\Project\Tranaformers\UserWalletLogTransformer;

class WalletController extends ApiController
{
    /**
     * 获取余额
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"我的钱包"},
     *     summary="余额",
     *     path="/api/user/wallet/money",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="number",
     *                      description="金额"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function money()
    {
        return $this->response->array(['data' => $this->user->getMoney()]);
    }

    /**
     * 获取佣金总额
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"我的钱包"},
     *     summary="获取佣金总额",
     *     path="/api/user/wallet/reward",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="number",
     *                      description="金额"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function orderRewardSum()
    {
        return $this->response->array(['data' => $this->user->walletLog(4)->sum('value')]);
    }

    /**
     * 获取佣金次数
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"我的钱包"},
     *     summary="获取佣金次数",
     *     path="/api/user/wallet/reward_count",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="integer",
     *                      description="数量"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function orderRewardCount()
    {
        return $this->response->array(['data' => $this->user->walletLog(4)->count('id')]);
    }

    /**
     * 钱包变动日志
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"我的钱包"},
     *     summary="钱包变动日志",
     *     path="/api/user/wallet/log",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Parameter(
     *         name="type",
     *         in="query",
     *         description="变动类型：1:充值，2:支付订单，3:提现，4:佣金，5:后台添加，6:提现失败返还",
     *         required=false,
     *         explode=true,
     *         @OA\Schema(
     *             type="integer",
     *             enum = {1,2,3,4,5,6},
     *         )
     *     ),
     *     @OA\Parameter(
     *         name="include",
     *         in="query",
     *         description="返回的日志包含对应对象",
     *         required=false,
     *         explode=true,
     *         @OA\Schema(
     *             type="string",
     *             enum = {"order"},
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/UserWalletLog"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function log()
    {
        \Validator::make(request()->all(), [
            'type' => 'nullable|in:1,2,3,4,5,6',
        ])->validate();

        $list = $this->user
        ->walletLog(request('type', false), request('money_type', 'money'))
        ->paginate(20);

        return $this->response->paginator($list, new UserWalletLogTransformer());
    }

    /**
     * 钱包某项数据之合
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"我的钱包"},
     *     summary="钱包某项数据总和",
     *     path="/api/user/wallet/sum",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Parameter(
     *         name="type",
     *         in="query",
     *         description="变动类型：1:充值，2:支付订单，3:提现，4:佣金，5:后台添加，6:提现失败返还",
     *         required=true,
     *         explode=true,
     *         @OA\Schema(
     *             type="integer",
     *             enum = {1,2,3,4,5,6},
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="number",
     *                      description="金额"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function sum()
    {
        \Validator::make(request()->all(), [
            'type' => 'required|in:1,2,3,4,5,6',
        ])->validate();

        $sum = $this->user
            ->walletLog(request('type', false), request('money_type', 'money'))
            ->sum('value');

        return $this->response->array(['data' => abs($sum)]);
    }

    /**
     * Update resource in storage.
     *
     * @param WalletRequest $request
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Put(
     *     tags={"我的钱包"},
     *     summary="重设支付密码",
     *     path="/api/user/wallet/pay_password",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="code",
     *                     type="string",
     *                     description="手机验证码",
     *                 ),
     *                 @OA\Property(
     *                     property="pay_password",
     *                     type="string",
     *                     description="支付密码"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function update(WalletRequest $request)
    {
        $this->user->wallet->updateOrCreate(['user_id' => $this->user->id], ['pay_password' => \Hash::make($request->pay_password)]);

        return $this->response->noContent()->setStatusCode(204);
    }
}
