<?php

namespace Modules\Project\Http\Controllers\Api;

use Modules\Base\Http\Controllers\Api\ApiController;
use Modules\Project\Http\Requests\FavoriteRequest;

class FavoriteController extends ApiController
{
    /**
     * 获取收藏列表
     *
     * @param FavoriteRequest $request
     *
     * @return mixed
     *
     * @OA\Get(
     *     tags={"收藏"},
     *     summary="收藏列表",
     *     path="/api/user/favorites",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Parameter(
     *         name="object",
     *         in="query",
     *         description="要查询的收藏对象类型",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *             default="goods"
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index(FavoriteRequest $request)
    {
        $object = $request->getClassName();
        $transformClass = $this->getObjectTransfromClass($object);

        return $this->response->collection($this->user->favorites($object)->get(), new $transformClass());
    }

    protected function getObjectTransfromClass($class)
    {
        $base = class_basename($class);
        $class = '\\Modules\\Project\\Tranaformers\\'.$base.'Transformer';

        return $class;
    }

    /**
     * 加入收藏
     *
     * @param FavoriteRequest $request
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Post(
     *     tags={"收藏"},
     *     summary="加入收藏",
     *     path="/api/user/favorites",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"object","id"},
     *                 @OA\Property(
     *                     property="object",
     *                     type="string",
     *                     example="goods",
     *                 ),
     *                 @OA\Property(
     *                     property="id",
     *                     type="integer",
     *                     example="1"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="成功"
     *      )
     * )
     */
    public function store(FavoriteRequest $request)
    {
        $class = $request->getClassName();
        $targets = $class::findOrFail($request->id);
        $this->user->favorite($targets);

        return $this->response->noContent()->setStatusCode(201);
    }

    /**
     * @param FavoriteRequest $request
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Delete(
     *     tags={"收藏"},
     *     summary="删除收藏",
     *     path="/api/user/favorites",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"object","id"},
     *                 @OA\Property(
     *                     property="object",
     *                     type="string",
     *                     example="goods",
     *                 ),
     *                 @OA\Property(
     *                     property="id",
     *                     type="integer",
     *                     example="1"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function destroy(FavoriteRequest $request)
    {
        $class = $request->getClassName();
        $targets = explode(',', $request->id);
        $this->user->unfavorite($targets, $class);

        return $this->response->noContent()->setStatusCode(204);
    }
}
