<?php

namespace Modules\Base\Http\Controllers\Api;

use Modules\Base\Http\Requests\UserRequest;
use Modules\Base\Services\SmsService;
use Modules\Base\Services\UserService;
use Modules\Base\Transformers\UserTransformer;

class UsersController extends ApiController
{
    /**
     * Store a newly created resource in storage.
     *
     * @param UserRequest $request
     * @param SmsService  $sms
     * @param UserService $userService
     *
     * @return \Dingo\Api\Http\Response $response
     *
     * @OA\Post(
     *     tags={"用户"},
     *     summary="用户注册接口",
     *     path="/api/base/users",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"phone","password", "code"},
     *                 @OA\Property(
     *                     property="phone",
     *                     ref="#/components/schemas/phone"
     *                 ),
     *                 @OA\Property(
     *                     property="code",
     *                     ref="#/components/schemas/phone_code"
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     ref="#/components/schemas/password"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/User")
     *              )
     *          )
     *      )
     * )
     */
    public function store(UserRequest $request, SmsService $sms, UserService $userService)
    {
        $sms->verifyCode($request->code, $sms::REGISTER, $request->phone);

        $request['password'] = \Hash::make($request->password);
        $userService->create($request->all());

        return $this->response->item($userService->model, new UserTransformer())
            ->setStatusCode(201);
    }

    /**
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"用户"},
     *     summary="当前登陆用户信息",
     *     path="/api/base/user",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/User")
     *              )
     *          )
     *      )
     * )
     */
    public function me()
    {
        return $this->response->item($this->user(), new UserTransformer());
    }

    /**
     * Update resource in storage.
     *
     * @param UserRequest $request
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Put(
     *     tags={"用户"},
     *     summary="更新用户信息",
     *     path="/api/base/user",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                 ),
     *                 @OA\Property(
     *                     property="email",
     *                     ref="#/components/schemas/email"
     *                 ),
     *                 @OA\Property(
     *                     property="file_id",
     *                     type="integer",
     *                     description="文件ID"
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     ref="#/components/schemas/password"
     *                 ),
     *                  @OA\Property(
     *                     property="oldpassword",
     *                     ref="#/components/schemas/password"
     *                 ),
     *                 @OA\Property(
     *                     property="phone",
     *                     ref="#/components/schemas/phone",
     *                     description="手机号",
     *                 ),
     *                 @OA\Property(
     *                     property="code",
     *                     ref="#/components/schemas/phone_code",
     *                     description="手机验证码，修改修改号时必须",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/User")
     *              )
     *          )
     *      )
     * )
     */
    public function update(UserRequest $request, UserService $user)
    {
        if ($request->phone) {
            $sms = app(SmsService::class);
            $sms->verifyCode($request->code, $sms::CHANGE_PHONE, $request->phone);
        }
        $user->update(\Auth::id(), $request->all());

        return $this->response->item($user->model, new UserTransformer());
    }

    /**
     * Update resource in storage.
     *
     * @param UserRequest $request
     *
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Put(
     *     tags={"用户"},
     *     summary="重设密码",
     *     path="/api/base/user/password",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="phone",
     *                     ref="#/components/schemas/phone"
     *                 ),
     *                 @OA\Property(
     *                     property="code",
     *                     ref="#/components/schemas/phone_code"
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     ref="#/components/schemas/password"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function password(UserRequest $request, UserService $userService, SmsService $sms)
    {
        $sms->verifyCode($request->code, $sms::FIND_PASSWORD, $request->phone);
        $user = $userService->model->where('phone', $request->phone)->firstOrFail();
        $userService->update($user->id, $request->only('password'));

        return $this->response->noContent();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     *
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }
}
