<?php

namespace Modules\Base\Http\Controllers\Api;

use Illuminate\Notifications\DatabaseNotification;
use Modules\Base\Notifications\TestNotify;
use Modules\Base\Transformers\NotificationTransformer;

class NotificationsController extends ApiController
{
    /**
     * @return mixed
     *
     * @OA\Get(
     *     tags={"用户"},
     *     summary="通知列表",
     *     path="/api/base/user/notifications",
     *     security={{"jwt_auth":{}}},
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Notification"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index()
    {
        $notifications = $this->user->notifications()->paginate(20);

        return $this->response->paginator($notifications, new NotificationTransformer());
    }

    /**
     * @return mixed
     *
     * @OA\Get(
     *     tags={"用户"},
     *     summary="查看通知",
     *     path="/api/base/user/notifications/{notification_id}",
     *     security={{"jwt_auth":{}}},
     *     @OA\Parameter(
     *         name="notification_id",
     *         in="path",
     *         description="通知ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Notification"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function show($notification_id)
    {
        $notification = DatabaseNotification::where('notifiable_id', $this->user->id)->findOrFail($notification_id);

        return $this->response->item($notification, new NotificationTransformer());
    }

    /**
     * @return mixed
     *
     * @OA\Get(
     *     tags={"用户"},
     *     summary="通知数据统计",
     *     path="/api/base/user/notifications/stats",
     *     security={{"jwt_auth":{}}},
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="count", type="integer", description="消息数量"),
     *                  @OA\Property(property="unread_count", type="integer", description="未读数量")
     *              )
     *          )
     *      )
     * )
     */
    public function stats()
    {
        $user = $this->user();

        return $this->response->array([
            'count' => $user->notifications()->count(),
            'unread_count' => $user->unreadNotifications()->count(),
            //'group_count'  => $user->notifications()->groupBy('notifiable_type')->count(),
            //'unread_unread_count' => $user->unreadNotifications()->groupBy('notifiable_type')->count(),
        ]);
    }

    /**
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Post(
     *     tags={"用户"},
     *     path="/api/base/user/notifications",
     *     summary="向登陆用户发送一条测试消息",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(response="201", description="成功")
     * )
     */
    public function test()
    {
        $this->user()->notify(new TestNotify(['id' => 111,'content' => '这里是内容', 'subject' => '这里是标题']));
        $notification->markAsRead();

        return $this->response->created();
    }

    /**
     * @return mixed
     *
     * @OA\Put(
     *     tags={"用户"},
     *     summary="所有通知标记为已读",
     *     path="/api/base/user/read/notifications",
     *     security={{"jwt_auth":{}}},
     *     @OA\Response(
     *          response="204",
     *          description="成功"
     *      )
     * )
     */
    public function read()
    {
        $this->user()->unreadNotifications()->update(['read_at' => now()]);

        return $this->response->noContent();
    }
}
