<?php

namespace Modules\Project\Http\Controllers\Api;

use Modules\Base\Http\Controllers\Api\ApiController;
use Modules\Project\Http\Requests\UserCouponRequest;
use Modules\Project\Services\CouponService;
use Modules\Project\Tranaformers\CouponTransformer;
use Modules\Project\Tranaformers\UserCouponTransformer;

class CouponController extends ApiController
{
    /**
     * 优惠券列表
     *
     * @param CouponService $couponService
     *
     * @return mixed
     *
     * @OA\Get(
     *     tags={"优惠券"},
     *     summary="优惠券列表",
     *     path="/api/coupons",
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Coupon"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index(CouponService $couponService)
    {
        return $this->response->collection($couponService->getList(), new CouponTransformer());
    }

    /**
     * 我的优惠券列表
     *
     * @param CouponService     $couponService
     * @param UserCouponRequest $request
     *
     * @return mixed
     *
     * @OA\Get(
     *     tags={"优惠券"},
     *     summary="我的优惠券",
     *     path="/api/user/coupons",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Parameter(
     *         name="status",
     *         in="query",
     *         description="优惠券状态",
     *         required=true,
     *         explode=true,
     *         @OA\Schema(
     *             type="string",
     *             enum = {"available", "expired", "used"},
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/UserCoupon"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function me(CouponService $couponService, UserCouponRequest $request)
    {
        return $this->response->collection($couponService->listOfUser($request->all()), new UserCouponTransformer());
    }

    /**
     * 领取优惠券
     *
     * @param CouponService     $couponService
     * @param UserCouponRequest $request
     *
     * @return mixed
     *
     * @OA\Post(
     *     tags={"优惠券"},
     *     summary="领取优惠券",
     *     path="/api/user/coupons",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"coupon_id"},
     *                 @OA\Property(
     *                     property="coupon_id",
     *                     type="integer",
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="成功"
     *      )
     * )
     */
    public function store(CouponService $couponService, UserCouponRequest $request)
    {
        $couponService->getCoupon($request->all());

        return $this->response->noContent()->setStatusCode(201);
    }
}
