<?php

namespace Modules\Project\Http\Controllers\Api;

use Modules\Base\Http\Controllers\Api\ApiController;
use Modules\Base\Services\SmsService;
use Modules\Project\Http\Requests\CardRequest;
use Modules\Project\Tranaformers\CardTransformer;

class CardController extends ApiController
{
    /**
     * @return mixed
     *
     * @OA\Get(
     *     tags={"银行卡"},
     *     summary="列表",
     *     path="/api/card",
     *     security={{"jwt_auth":{}}},
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Card"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index()
    {
        $cards = $this->user->card()->paginate(20);

        return $this->response->paginator($cards, new CardTransformer());
    }

    /**
     * @return mixed
     *
     * @OA\Post(
     *     tags={"银行卡"},
     *     summary="添加银行卡",
     *     path="/api/card",
     *     description="发送短信前需要验证图形验证码",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","number","branch","phone","code"},
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                 ),
     *                 @OA\Property(
     *                     property="number",
     *                     type="string",
     *                 ),
     *                 @OA\Property(
     *                     property="branch",
     *                     type="string",
     *                 ),
     *                 @OA\Property(
     *                     property="phone",
     *                     type="string",
     *                 ),
     *                 @OA\Property(
     *                     property="code",
     *                     type="string",
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Card")
     *              )
     *          )
     *      )
     * )
     */
    public function store(CardRequest $request, SmsService $smsService)
    {
        $smsService->verifyCode(request()->get('code'), $smsService::ADD_CARD, $request->phone);
        $card = $this->user->card()->create($request->all());

        return $this->response->item($card, new CardTransformer())
            ->setStatusCode(201);
    }

    /**
     * @return mixed
     *
     * @OA\Put(
     *     tags={"银行卡"},
     *     summary="设为默认银行卡",
     *     path="/api/card/{card_id}",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Parameter(
     *         name="card_id",
     *         in="path",
     *         description="银行卡ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"is_default"},
     *                 @OA\Property(
     *                     property="is_default",
     *                     type="integer",
     *                     enum={1},
     *                     example=1,
     *                 ),
     *
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Card")
     *              )
     *          )
     *      )
     * )
     */
    public function update($card_id, CardRequest $request)
    {
        $card = $this->user->card()->findOrFail($card_id);

        $card->update($request->all());

        return $this->response->item($card, new CardTransformer())
            ->setStatusCode(200);
    }

    /**
     * @param $card_id
     *
     * @return mixed
     * @OA\Delete(
     *     tags={"银行卡"},
     *     summary="删除银行卡",
     *     path="/api/card/{card_id}",
     *     security={{"jwt_auth":{}}},
     *     @OA\Parameter(
     *         name="card_id",
     *         in="path",
     *         description="银行卡ID",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\Response(
     *          response="204",
     *          description="",
     *      )
     * )
     */
    public function destroy($card_id)
    {
        $card = $this->user->card()->findOrFail($card_id);
        $card->delete();

        return $this->response->noContent()->setStatusCode(204);
    }
}
