<?php

namespace Modules\Base\Http\Controllers\Api;

use Illuminate\Support\Facades\Event;
use Modules\Base\Entities\Activity;
use Modules\Base\Entities\Enrollment;
use Modules\Base\Events\EnrollmentCreated;
use Modules\Base\Http\Requests\EnrollmentRequest;
use Modules\Base\Transformers\EnrollmentTransformer;

class EnrollmentController extends ApiController
{
    /**
     * 已报名用户信息列表
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"报名"},
     *     summary="报名信息",
     *     path="/api/enrollments",
     *     @OA\Parameter(
     *         name="activity_id",
     *         in="query",
     *         description="活动ID",
     *         required=true,
     *         explode=true,
     *         @OA\Schema(
     *             type="number"
     *         )
     *     ),
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(
     *                      property="data",
     *                      type="array",
     *                      @OA\Items(ref="#/components/schemas/Enrollment"),
     *                      description="列表"
     *                  ),
     *              )
     *          )
     *      )
     * )
     */
    public function index()
    {
        //$list = Enrollment::with('user')->latest()->paginate();
        $activity = Activity::findOrFail(request()->activity_id);
        return $this->response->paginator($activity->enrollments()->paginate(), new EnrollmentTransformer());
    }

    /**
     * 个人报名信息
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"报名"},
     *     summary="个人报名信息",
     *     path="/api/user/enrollment",
     *     @OA\Parameter(
     *         name="activity_id",
     *         in="path",
     *         description="活动ID",
     *         required=true,
     *         @OA\Schema(
     *             type="integer",
     *         )
     *     ),
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Enrollment")
     *              )
     *          )
     *      )
     * )
     */
    public function me()
    {
        return $this->response->item($this->user()->enrollment, new EnrollmentTransformer(true));
    }
    /**
     * 个人报名信息
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Get(
     *     tags={"报名"},
     *     summary="已报名列表",
     *     path="/api/user/enrollment",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="成功",
     *          @OA\MediaType(
     *             mediaType="application/json",
     *              @OA\Schema(type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Enrollment")
     *              )
     *          )
     *      )
     * )
     */
    public function meEnrollments()
    {
        return $this->response->paginator($this->user()->enrollments()->paginate(20), new EnrollmentTransformer(true));
    }

    /**
     * 报名
     * @param EnrollmentRequest $request
     * @return \Dingo\Api\Http\Response
     *
     * @OA\Post(
     *     tags={"报名"},
     *     summary="提交报名信息",
     *     path="/api/enrollments",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\RequestBody(
     *         required=true,
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"name","mobile", "company","activity_id"},
     *                 @OA\Property(
     *                     property="activity_id",
     *                     type="number",
     *                     description="活动ID"
     *                 ),
     *                 @OA\Property(
     *                     property="name",
     *                     type="string",
     *                     description="姓名"
     *                 ),
     *                @OA\Property(
     *                     property="mobile",
     *                     type="number",
     *                     description="手机号"
     *                 ),
     *                 @OA\Property(
     *                     property="company",
     *                     type="string",
     *                     description="公司"
     *                 ),
     *                 @OA\Property(
     *                     property="recommender",
     *                     type="string",
     *                     description="推荐人姓名"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="成功"
     *      )
     * )
     */
    public function store(EnrollmentRequest $request)
    {
        $enrollment = $this->user()->enrollment()->create($request->all());
        event(new EnrollmentCreated($enrollment));
        return $this->response->created();
    }
}
