<?php

namespace Modules\Base\Http\Controllers\Api;

use Illuminate\Support\Facades\Auth;
use Modules\Base\Http\Requests\AuthorizationRequest;
use Modules\Base\Http\Requests\OpenloginRequest;
use Modules\Base\Services\OpenloginService;
use Modules\Base\Transformers\AccessTokenTransformer;
use Modules\Base\Transformers\UserTransformer;

class AuthorizationsController extends ApiController
{
    /**
     * 登陆
     *
     * @param AuthorizationRequest $request
     *
     * @return mixed
     *
     * @OA\Post(
     *     tags={"用户"},
     *     summary="用户登陆",
     *     path="/api/base/authorizations",
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 required={"phone","password"},
     *                 @OA\Property(
     *                     property="phone",
     *                     ref="#/components/schemas/phone"
     *                 ),
     *                 @OA\Property(
     *                     property="password",
     *                     ref="#/components/schemas/password"
     *                 ),
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="登陆成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Token")
     *              )
     *          )
     *      )
     * )
     */
    public function store(AuthorizationRequest $request)
    {
        if (!$token = Auth::guard('api')->attempt($request->only(['phone', 'password']))) {
            return $this->response->errorUnauthorized('手机号或密码错误');
        }

        return $this->response->item(auth('api')->user(), new AccessTokenTransformer())->setStatusCode(201);
    }

    /**
     * 第三方登陆
     *
     * @param $type
     * @param OpenloginRequest $request
     * @param OpenloginService $openlogin
     *
     * @return $token
     *
     * @OA\Post(
     *     tags={"用户"},
     *     summary="第三方账号登陆",
     *     path="/api/base/authorizations/openlogin/{type}",
     *     @OA\Parameter(
     *         name="type",
     *         in="path",
     *         description="第三方登陆类型",
     *         required=true,
     *         @OA\Schema(
     *             type="string",
     *         )
     *     ),
     *     @OA\RequestBody(
     *         @OA\MediaType(
     *             mediaType="application/x-www-form-urlencoded",
     *             @OA\Schema(
     *                 type="object",
     *                 @OA\Property(
     *                     property="code",
     *                     type="string"
     *                 )
     *             )
     *         )
     *     ),
     *     @OA\Response(
     *          response="201",
     *          description="登陆成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Token")
     *              )
     *          )
     *      )
     * )
     */
    public function openlogin($type, OpenloginRequest $request, OpenloginService $openlogin)
    {
        $user = $openlogin->login($type, $request->all());

        return $this->response->item($user, new AccessTokenTransformer())->setStatusCode(201);
    }

    /**
     * @return mixed
     *
     * @OA\Put(
     *     tags={"用户"},
     *     summary="刷新TOKEN",
     *     path="/api/base/authorizations/current",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="200",
     *          description="刷新成功",
     *          @OA\MediaType(
     *              mediaType="application/json",
     *              @OA\Schema(
     *                  type="object",
     *                  @OA\Property(property="data", type="object", ref="#/components/schemas/Token")
     *              )
     *          )
     *      )
     * )
     */
    public function update()
    {
        $token = \Auth::guard('api')->refresh();

        return $this->respondWithToken($token);
    }

    /**
     * @return mixed
     *
     * @OA\Delete(
     *     tags={"用户"},
     *     summary="用户登出",
     *     path="/api/base/authorizations/current",
     *     security={
     *         {"jwt_auth": {}}
     *     },
     *     @OA\Response(
     *          response="204",
     *          description="登出成功"
     *      )
     * )
     */
    public function destroy()
    {
        \Auth::logout();

        return $this->response->noContent()->setStatusCode(200);
    }
}
