<?php

namespace App\Admin\Controllers;

use App\Http\Controllers\Controller;
use Encore\Admin\Controllers\HasResourceActions;
use Encore\Admin\Form;
use Encore\Admin\Grid;
use Encore\Admin\Layout\Content;
use Encore\Admin\Show;
use App\models\Article;

class ArticleController extends Controller
{
    use HasResourceActions;

    /**
     * Index interface.
     *
     * @param Content $content
     * @return Content
     */
    public function index(Content $content)
    {
        return $content
            ->header('文章管理')
            ->description('列表')
            ->breadcrumb(
                ['text' => '文章管理', 'url' => '/admin/article'],
                ['text' => '文章列表']
            )
            ->body($this->grid());
    }

    /**
     * Edit interface.
     *
     * @param mixed   $id
     * @param Content $content
     * @return Content
     */
    public function edit($id, Content $content)
    {
        return $content
            ->header('编辑文章')
            ->description('修改')
            ->breadcrumb(
                ['text' => '文章管理', 'url' => '/admin/article'],
                ['text' => '编辑文章']
            )
            ->body($this->form()->edit($id));
    }

    /**
     * Create interface.
     *
     * @param Content $content
     * @return Content
     */
    public function create(Content $content)
    {
        return $content
            ->header('新增文章')
            ->description('添加')
            ->breadcrumb(
                ['text' => '文章管理', 'url' => '/admin/article'],
                ['text' => '新增文章']
            )
            ->body($this->form());
    }

    /**
     * Make a grid builder.
     *
     * @return Grid
     */
    protected function grid()
    {
        $grid = new Grid(new Article);

        $grid->id('ID')->sortable();
        $grid->title()->editable();
        $grid->views('浏览量')->sortable();
        $grid->likes('点赞量')->sortable();
        $grid->collects('收藏量')->sortable();
        $grid->column('is_quintessence','是否精华')->select(['否','精华'])->sortable();
        $grid->column('is_top','是否置顶')->select(['否','置顶'])->sortable();
        $grid->status('状态')->select(['待审核','显示','隐藏'])->sortable();
        $grid->updated_at('更新时间')->sortable();

        $grid->actions(function ($actions) {
            $actions->disableView();
        });

        $grid->expandFilter();
        $grid->disableExport();

        $grid->filter(function($filter){
            $filter->column(1/2, function ($filter) {
                $filter->like('title', '标题');
                $filter->between('updated_at','更新时间')->datetime();
            });
            $filter->column(1/2, function ($filter) {
                $filter->equal('is_quintessence','是否精华')->select(['否','是']);
                $filter->equal('is_top','是否置顶')->select(['否','是']);
                $filter->equal('status','状态')->select(['待审核','显示','隐藏']);
            });
        });

        return $grid;
    }

    /**
     * Make a form builder.
     *
     * @return Form
     */
    protected function form()
    {
        $form = new Form(new Article);

        $form->hidden('id', 'ID');
        $form->image('cover')->move('article_cover')->uniqueName();
        $form->text('title','标题')->rules('required',['required'=>'请输入标题']);
        $form->text('author','作者');
        $form->textarea('keywords','关键词')->help('作用：用于百度检索，词之间用逗号分隔开');
        $form->textarea('description','描述')->help('作用：用于百度检索');
        $form->textarea('abstract','摘要');
        $form->ckeditor('content','内容详情');
        $form->tags('labels','标签');
        $form->number('sort','排序');
        $form->radio('status','状态')->options(['待审核','显示','隐藏'])->default(0);
        $form->radio('is_quintessence','是否精华')->options(['否','是'])->default(0);
        $form->radio('is_top','是否置顶')->options(['否','是'])->default(0);

        $form->tools(function (Form\Tools $tools) {
            $tools->disableDelete();    // 去掉`删除`按钮
            $tools->disableView();      // 去掉`查看`按钮
        });

        $form->footer(function ($footer) {
            $footer->disableViewCheck();        // 去掉`查看`checkbox
            $footer->disableEditingCheck();    // 去掉`继续编辑`checkbox
            $footer->disableCreatingCheck();    // 去掉`继续编辑`checkbox
        });

        return $form;
    }

    /**
     * Make a show builder.
     *
     * @param mixed   $id
     * @return Show
     */
    protected function detail($id)
    {
        $show = new Show(Article::findOrFail($id));

        $show->id('ID');
        $show->created_at('Created at');
        $show->updated_at('Updated at');

        return $show;
    }
}
